/*
 * RootItem.cpp
 *
 *  Created on: 29.10.2009
 *      Author: stefan.detter
 */

#include "RootItem.h"

#include <QFont>
#include <QBrush>

#include "ReaderEntry.h"
#include "TagEntry.h"

#include "../../application/Reader.h"

#include "../../Settings.h"

RootItem::RootItem(TreeItem* parent)
	: TreeItem(parent)
{
	prepareInformation ( );

	m_updateTimer = new QTimer(this);
	m_updateTimer->setSingleShot(true);
	connect(m_updateTimer, SIGNAL(timeout()), this, SLOT(update()));
	m_updateTimer->start(Settings::d->refreshRate());
}

RootItem::~RootItem()
{
	m_updateTimer->stop();
}

void RootItem::updateDynamicInfo()
{
	updateInformation();
	TreeItem::updateDynamicInfo();
}

void RootItem::clearTags()
{
	m_readUniqueTags.clear();
	m_readReaderTags.clear();
	updateInformation();
	TreeItem::clearTags();
}

QList<QString> RootItem::readerList ( ) const
{
	return m_readerIndex.keys();
}

QList<QString> RootItem::tagList ( ) const
{
	QSet<QString> set;
	foreach(QString reader, m_readerIndex.keys())
	{
        set += m_readerIndex[reader]->uniqueTagIds();
	}

	return set.toList();
}

QList<QString> RootItem::tagListOfReader( QString reader ) const
{
	if(!m_readerIndex.contains(reader))
		return QList<QString>();

    return m_readerIndex[reader]->uniqueTagIds().toList();
}

Reader* RootItem::getReader ( QString readerId )
{
	ReaderEntry* r = dynamic_cast<ReaderEntry*>(m_readerIndex.value(readerId));
	return r->reader();
}

ReaderEntry* RootItem::addReader ( Reader* ri )
{
	if(m_readerIndex.contains(ri->readerId()))
		removeReader(ri->readerId());

	connect(ri, 	SIGNAL(tagEvent(const TagEvent& )),
			this, 	  SLOT(tagEvent(const TagEvent& )));

	emit beginInsertRows(this, m_readerIndex.size(), m_readerIndex.size() );

	ReaderEntry* r = new ReaderEntry(this, ri);

	connectChild(r);

	connect(r, 		SIGNAL(aboutToBeRemoved(TreeItem*)),
			this, 	SLOT(itemAboutToBeRemoved(TreeItem*)) );

	m_children.append(r);
	m_readerIndex.insert(r->readerId(), r);

	emit endInsertRows();

	updateInformation();

	return r;
}

void RootItem::removeReader ( QString readerId )
{
	ReaderEntry* r = m_readerIndex.value(readerId);

	r->clearTags();

	emit aboutToBeRemoved(r);

	emit beginRemoveRows( this, m_children.indexOf(r), m_children.indexOf(r) );
	m_children.removeAt(m_children.indexOf(r));
	m_readerIndex.remove(readerId);
	emit endRemoveRows();

	disconnectChild(r);
	r->deleteLater();

	updateInformation();
}

void RootItem::removeOfflineReader ( )
{
	foreach(QString readerId, m_readerIndex.keys())
	{
		ReaderEntry* r = m_readerIndex.value(readerId);

		if(r->isOffline())
		{
			removeReader(readerId);
		}
	}
}


void RootItem::tagEvent( const TagEvent& event)
{
	m_readUniqueTags.insert(event.tagType() + event.tagId());
	m_readReaderTags.insert(event.readerId() + event.tagType() + event.tagId());
	m_shownUniqueTags.insert(event.tagType() + event.tagId());
	m_shownReaderTags.insert(event.readerId() + event.tagType() + event.tagId());
}

void RootItem::itemAboutToBeRemoved ( TreeItem* t )
{
	TagEntry* te = dynamic_cast<TagEntry*>(t);
	if(te == 0)
		return;

	m_shownUniqueTags.remove(te->tagType() + te->tagId());
	m_shownReaderTags.remove(te->reader()->readerId() + te->tagType() + te->tagId());
}

void RootItem::update (  )
{
	updateDynamicInfo();
	m_updateTimer->start(Settings::d->refreshRate());
}
